﻿using System;
using System.Windows.Forms;
using System.Runtime.InteropServices;

namespace ExampleUSBTL50
{
    public partial class Form1 : Form
    {
        // DLL

        // Init is used to find the correct COM port and put the device into a known state
        [DllImport(_DLL_PATH, CallingConvention = CallingConvention.Cdecl)]
        public static extern int Init();

        // An alternate Init to use the raw port number when initilizing
        [DllImport(_DLL_PATH, CallingConvention = CallingConvention.Cdecl)]
        public static extern int InitByPort(int PortNumber);

        // Sets a segment with the desired color (Segment is used on multiple segment stacks, if using a single segment, just set at 0)
        [DllImport(_DLL_PATH, CallingConvention = CallingConvention.Cdecl)]
        public static extern int SetSegmentSolid(int Segment, int Color);

        // Sets a segment off (Segment is used on multiple segment stacks, if using a single segment, just set at 0)
        [DllImport(_DLL_PATH, CallingConvention = CallingConvention.Cdecl)]
        public static extern int SetSegmentOff(int Segment);

        // Can be used to cycle the device in case of error.  Will take time for USB to re-enum!
        [DllImport(_DLL_PATH, CallingConvention = CallingConvention.Cdecl)]
        public static extern int Deinit();

        // Returns the version of the dll
        [DllImport(_DLL_PATH, CallingConvention = CallingConvention.Cdecl)]
        public static extern UInt16 GetDllVersion();

        // Used in more advanced applications, see example by double clicking background of form
        [DllImport(_DLL_PATH, CallingConvention = CallingConvention.Cdecl)]
        public static extern int SetSegmentAdvanced(int Segment, byte[] Data);

        // On models with an audible segment, this controls the sound
        [DllImport(_DLL_PATH, CallingConvention = CallingConvention.Cdecl)]
        public static extern int SetAudible(int AlarmType);

        public Form1()
        {
            InitializeComponent();
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            // Load the CBs
            foreach(string Color in _COLORS)
            {
                ColorNumberCombo.Items.Add(Color);
            }
            ColorNumberCombo.SelectedIndex = 0;
            SegmentNumberTb.Text = "1";  // control will be a 1-based index
            UInt16 Version = 0;
            try
            {
                Version = GetDllVersion();
            }
            catch
            {
                MessageBox.Show("Could not load DLL, please check driver.");
                this.Close();
            }
            // Unpack BCD
            Byte Major = (Byte)(Version / 256);
            Byte Minor = (Byte)(Version);
            this.Text += " " + GuiVersion;
            this.Text += " (DLL " + Major.ToString() + "." + Minor.ToString() + ")";
        }

        public static readonly string[] _COLORS = { "Green", "Red", "Orange", "Amber", "Yellow", "Lime Green", "Spring Green", "Cyan", "Sky Blue", "Blue", "Violet", "Magenta", "Rose", "White"};

        private void ConnectBtn_Click(object sender, EventArgs e)
        {
            int Status = Init();
            if(Status > 0)
            {
                PortNumberTb.Text = Status.ToString();
            }
            else
            {
                PortNumberTb.Text = "";
                ShowErrorMessage(Status);
            }
        }

        private void SetSegmentOffBtn_Click(object sender, EventArgs e)
        {
            // Should check for valid string and number!
            int Segment = int.Parse(SegmentNumberTb.Text) - 1;  // control is 1-based, but library is 0-based index
            int Status = SetSegmentOff(Segment);
            if (Status != (int)CommReturnValue.SUCCESS)
            {
                ShowErrorMessage(Status);
            }
        }

        // Colors available
        public enum ColorType
        {
            COLOR_GREEN,
            COLOR_RED,
            COLOR_ORANGE,
            COLOR_AMBER,
            COLOR_YELLOW,
            COLOR_LIME_GREEN,
            COLOR_SPRING_GREEN,
            COLOR_CYAN,
            COLOR_SKY_BLUE,
            COLOR_BLUE,
            COLOR_VIOLET,
            COLOR_MAGENTA,
            COLOR_ROSE,
            COLOR_WHITE,
            COLOR_CUSTOM1,
            COLOR_CUSTOM2,
            COLOR_MAX,
        };

        private void SteadyColorBtn_Click(object sender, EventArgs e)
        {
            // Should check for valid string and number!
            int Segment = int.Parse(SegmentNumberTb.Text) - 1;  // control is 1-based, but library is 0-based index
            int ColorNumber = ColorNumberCombo.SelectedIndex;
            int Status = SetSegmentSolid(Segment, ColorNumber);
            if (Status != (int)CommReturnValue.SUCCESS)
            {
                ShowErrorMessage(Status);
            }
        }

        private const string _DLL_PATH = "Tl50UsbLibraryx64.dll";

        private void CloseBtn_Click(object sender, EventArgs e)
        {
            Deinit();
            PortNumberTb.Text = "";
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            try
            {
                Deinit();
            }
            catch
            {

            }
            
        }

        private const string GuiVersion = "1.0";

        enum CommReturnValue
        {
            SUCCESS = 0,
            FAILED_PORT_NOT_FOUND = -1,
            FAILED_PORT_OPEN = -2,
            FAILED_WRITE = -3,
            FAILED_READ = -4,
            FAILED_CHECKSUM = -5,
            FAIL_WITH_NACK = -6,
            FAILED_NO_INIT = -7,
        };

        public static void ShowErrorMessage(int Error)
        {
            string ErrorMessage;
            switch (Error)
            {
                case (int)CommReturnValue.FAILED_PORT_NOT_FOUND:
                    ErrorMessage = "Port was not found.";
                    break;
                case (int)CommReturnValue.FAILED_PORT_OPEN:
                    ErrorMessage = "Could not open port.";
                    break;
                case (int)CommReturnValue.FAILED_WRITE:
                    ErrorMessage = "Could not write.";
                    break;
                case (int)CommReturnValue.FAILED_READ:
                    ErrorMessage = "Could not read.";
                    break;
                case (int)CommReturnValue.FAILED_CHECKSUM:
                    ErrorMessage = "Checksum error.";
                    break;
                case (int)CommReturnValue.FAIL_WITH_NACK:
                    ErrorMessage = "Device NACKed.";
                    break;
                case (int)CommReturnValue.FAILED_NO_INIT:
                    ErrorMessage = "Port was not initialized.";
                    break;
                default:
                    ErrorMessage = "Unknown error";
                    break;
            }
            MessageBox.Show(ErrorMessage);
        }

        private void PortNumberTb_DoubleClick(object sender, EventArgs e)
        {
            int Status = 0;
            string UserPort = PromptUserForInput();
            if (UserPort == "")
            {
                PortNumberTb.Text = "";
            }
            else
            {
                int Port = 0;
                if (int.TryParse(UserPort, out Port) && Port > 0)
                {
                    Status = InitByPort(Port);
                }
                else
                {
                    Status = (int)CommReturnValue.FAILED_PORT_NOT_FOUND;
                }
                if (Status > 0)
                {
                    PortNumberTb.Text = Status.ToString();
                }
                else
                {
                    PortNumberTb.Text = "";
                    ShowErrorMessage(Status);
                }
            }
        }

        private string PromptUserForInput()
        {
            const string PromptText = "Port Number?";
            Form prompt = new Form()
            {
                Width = 100,
                Height = 150,
                FormBorderStyle = FormBorderStyle.FixedDialog,
                Text = "Enter Com Port",
                StartPosition = FormStartPosition.CenterScreen
            };
            Label textLabel = new Label() { Left = 25, Top = 20, Text = PromptText};
            TextBox textBox = new TextBox() { Left = 20, Top = 40, Width = 80 };
            Button confirmation = new Button() { Text = "Ok", Left = 20, Width = 80, Top = 70, DialogResult = DialogResult.OK };
            confirmation.Click += (sender, e) => { prompt.Close(); };
            prompt.Controls.Add(textBox);
            prompt.Controls.Add(confirmation);
            prompt.Controls.Add(textLabel);
            prompt.AcceptButton = confirmation;

            return prompt.ShowDialog() == DialogResult.OK ? textBox.Text : "";
        }

        private void AdvancedSettingsBtn_Click(object sender, EventArgs e)
        {
            AdvancedOptions Options = new AdvancedOptions();
            Options.StartPosition = FormStartPosition.CenterParent;
            Options.ShowDialog(this);
        }


	}
}
